<?php
header('Content-Type: application/json');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');

session_start();
require '../config/config.php'; // Conexión a MySQL optimizada
// 1. Validación de autenticación y permisos mejorada
if (!isset($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'No autorizado', 'code' => 'UNAUTHORIZED']);
    exit;
}


// 3. Validación de CSRF Token (si usas formularios tradicionales)
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(403);
    echo json_encode(['error' => 'Solicitud no permitida', 'code' => 'CSRF_DETECTED']);
    exit;
}



// 5. Conexión segura a la base de datos
try {

    // Configurar charset para prevenir inyección SQL
    $conexion->set_charset("utf8mb4");

    // 6. Validación estricta de entrada JSON
    $json_data = file_get_contents('php://input');
    if (empty($json_data)) {
        throw new Exception("Datos JSON no proporcionados", 400);
    }

    $data = json_decode($json_data, true, 10); // Profundidad máxima 10
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("JSON inválido: " . json_last_error_msg(), 400);
    }

    // 7. Validación exhaustiva de datos
    $errors = [];
    $validated = [
        'cuestionario_id' => filter_var($data['cuestionario_id'] ?? null, FILTER_VALIDATE_INT, [
            'options' => ['min_range' => 1]
        ]),
        'texto_pregunta' => isset($data['texto_pregunta']) ? trim($data['texto_pregunta']) : null,
        'puntaje' => filter_var($data['puntaje'] ?? null, FILTER_VALIDATE_FLOAT, [
            'options' => ['min_range' => 0]
        ]),
        'opciones' => []
    ];

    if (!$validated['cuestionario_id']) {
        $errors[] = "ID de cuestionario inválido o faltante";
    }
    if (empty($validated['texto_pregunta'])) {
        $errors[] = "El texto de la pregunta no puede estar vacío";
    } elseif (strlen($validated['texto_pregunta']) > 500) {
        $errors[] = "El texto de la pregunta excede el límite de 500 caracteres";
    }
    if ($validated['puntaje'] === false) {
        $errors[] = "El puntaje debe ser un número mayor o igual a 0";
    }

    // Validar opciones
    if (!isset($data['opciones']) || !is_array($data['opciones']) || count($data['opciones']) < 2) {
        $errors[] = "Debe haber al menos dos opciones de respuesta";
    } else {
        $respuesta_correcta_encontrada = false;
        foreach ($data['opciones'] as $i => $opcion) {
            $texto_opcion = isset($opcion['texto_opcion']) ? trim($opcion['texto_opcion']) : null;
            $es_correcta = isset($opcion['es_correcta']) ? (int)$opcion['es_correcta'] : null;

            if (empty($texto_opcion)) {
                $errors[] = "Opción #" . ($i + 1) . ": El texto no puede estar vacío";
            } elseif (strlen($texto_opcion) > 200) {
                $errors[] = "Opción #" . ($i + 1) . ": El texto excede el límite de 200 caracteres";
            }

            if (!in_array($es_correcta, [0, 1], true)) {
                $errors[] = "Opción #" . ($i + 1) . ": Valor de 'es_correcta' inválido (debe ser 0 o 1)";
            }

            if ($es_correcta === 1) {
                $respuesta_correcta_encontrada = true;
            }

            $validated['opciones'][] = [
                'texto_opcion' => $texto_opcion,
                'es_correcta' => $es_correcta
            ];
        }

        if (!$respuesta_correcta_encontrada) {
            $errors[] = "Debe haber al menos una opción marcada como correcta";
        }
    }

    if (!empty($errors)) {
        throw new Exception(implode("\n", $errors), 400);
    }

    // 8. Iniciar transacción para integridad de datos
    $conexion->begin_transaction();

    try {
        // 9. Insertar pregunta con consulta preparada
        $sql_pregunta = "INSERT INTO preguntas (cuestionario_id, texto_pregunta, puntaje) VALUES (?, ?, ?)";
        $stmt_pregunta = $conexion->prepare($sql_pregunta);
        if (!$stmt_pregunta) {
            throw new Exception("Error al preparar consulta: " . $conexion->error, 500);
        }

        $creado_por = $_SESSION['usuario_id'];
        $stmt_pregunta->bind_param(
            "isd",
            $validated['cuestionario_id'],
            $validated['texto_pregunta'],
            $validated['puntaje']
        );

        if (!$stmt_pregunta->execute()) {
            throw new Exception("Error al guardar la pregunta: " . $stmt_pregunta->error, 500);
        }

        $pregunta_id = $stmt_pregunta->insert_id;
        $stmt_pregunta->close();

        // 10. Insertar opciones con lote preparado
        $sql_opcion = "INSERT INTO opciones_pregunta (pregunta_id, texto_opcion, es_correcta) VALUES (?, ?, ?)";
        $stmt_opcion = $conexion->prepare($sql_opcion);
        if (!$stmt_opcion) {
            throw new Exception("Error al preparar consulta: " . $conexion->error, 500);
        }

        foreach ($validated['opciones'] as $orden => $opcion) {

            $stmt_opcion->bind_param(
                "isi",
                $pregunta_id,
                $opcion['texto_opcion'],
                $opcion['es_correcta']
            );

            if (!$stmt_opcion->execute()) {
                throw new Exception("Error al guardar opción: " . $stmt_opcion->error, 500);
            }
        }
        $stmt_opcion->close();

        // 12. Confirmar transacción
        $conexion->commit();

        // 13. Respuesta exitosa
        echo json_encode([
            'success' => true,
            'message' => 'Pregunta y opciones guardadas exitosamente',
            'data' => [
                'pregunta_id' => $pregunta_id,
                'num_opciones' => count($validated['opciones'])
            ]
        ]);
    } catch (Exception $e) {
        $conexion->rollback();
        throw $e; // Re-lanzar para manejo externo
    }
} catch (Exception $e) {
    // 14. Manejo centralizado de errores
    $status_code = $e->getCode() >= 400 && $e->getCode() < 600 ? $e->getCode() : 500;
    http_response_code($status_code);

    error_log("Error en guardar_pregunta.php: " . $e->getMessage());

    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'code' => $status_code === 400 ? 'VALIDATION_ERROR' : 'SERVER_ERROR'
    ]);
} finally {
    // 15. Limpieza segura
    if (isset($conexion)) {
        $conexion->close();
    }
}