<?php
header('Content-Type: application/json');
require '../config/config.php'; // Conexión a MySQL optimizada
$conexion->options(MYSQLI_OPT_CONNECT_TIMEOUT, 10);
session_start(); // Iniciar sesión si aún no está iniciada

// Verificar si el usuario está autenticado (ejemplo básico)
if (!isset($_SESSION['usuario_id'])) {
    http_response_code(401); // No autorizado
    echo json_encode(['error' => 'No autorizado']);
    exit;
}
// Validaciones iniciales (método, autenticación, CSRF)
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Método no permitido']);
    exit;
}


// Obtener y validar datos JSON
$data = json_decode(file_get_contents('php://input'), true);
if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Datos JSON inválidos']);
    exit;
}

// Validación de campos requeridos
$requiredFields = ['curso_id', 'titulo', 'descripcion', 'nivel', 'cuestionario_id'];
$errors = [];

foreach ($requiredFields as $field) {
    if (empty($data[$field])) {
        $errors[$field] = "El campo $field es obligatorio";
    }
}

// Validaciones específicas
if (!filter_var($data['curso_id'], FILTER_VALIDATE_INT)) {
    $errors['curso_id'] = "ID de curso inválido";
}

if (!filter_var($data['cuestionario_id'], FILTER_VALIDATE_INT)) {
    $errors['cuestionario_id'] = "ID de cuestionario inválido";
}

if (!filter_var($data['nivel'], FILTER_VALIDATE_INT, ['options' => ['min_range' => 1, 'max_range' => 100]])) {
    $errors['nivel'] = "El orden debe ser un número entre 1 y 10";
}

// Sanitizar datos
$cursoId = filter_var($data['curso_id'], FILTER_SANITIZE_NUMBER_INT);
$titulo = htmlspecialchars(trim($data['titulo']), ENT_QUOTES, 'UTF-8');
$descripcion = htmlspecialchars(trim($data['descripcion']), ENT_QUOTES, 'UTF-8');
$orden = filter_var($data['nivel'], FILTER_SANITIZE_NUMBER_INT);
$cuestionarioId = filter_var($data['cuestionario_id'], FILTER_SANITIZE_NUMBER_INT);

// Validar que el título no exista para este cuestionario
if (empty($errors)) {
    $stmtCheckTitle = $conexion->prepare("
        SELECT COUNT(*) as existe 
        FROM titulos_temarios tt
        JOIN titulos_temarios_cuestionarios ttc ON tt.ID_Titulo = ttc.ID_Titulo
        WHERE ttc.cuestionario_id = ? AND tt.nom_titulo_temario = ?
    ");
    $stmtCheckTitle->bind_param("is", $cuestionarioId, $titulo);
    $stmtCheckTitle->execute();
    $result = $stmtCheckTitle->get_result();
    $row = $result->fetch_assoc();

    if ($row['existe'] > 0) {
        $errors['titulo'] = "Ya existe un título con este nombre en el cuestionario seleccionado";
    }
    $stmtCheckTitle->close();
}

if (!empty($errors)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'errors' => $errors]);
    exit;
}

// Iniciar transacción
$conexion->begin_transaction();

try {
    // 1. Insertar en titulos_temarios
    $stmtInsertTitulo = $conexion->prepare("
        INSERT INTO titulos_temarios (curso_id, nom_titulo_temario, descripcion_temario, orden) 
        VALUES (?, ?, ?, ?)
    ");
    $stmtInsertTitulo->bind_param("issi", $cursoId, $titulo, $descripcion, $orden);

    if (!$stmtInsertTitulo->execute()) {
        throw new Exception("Error al insertar en titulos_temarios");
    }

    $idTitulo = $conexion->insert_id;
    $stmtInsertTitulo->close();

    // 2. Verificar que el cuestionario existe
    $stmtCheckCuestionario = $conexion->prepare("
        SELECT cuestionario_id FROM cuestionarios WHERE cuestionario_id = ?
    ");
    $stmtCheckCuestionario->bind_param("i", $cuestionarioId);
    $stmtCheckCuestionario->execute();

    if ($stmtCheckCuestionario->get_result()->num_rows === 0) {
        throw new Exception("El cuestionario especificado no existe");
    }
    $stmtCheckCuestionario->close();

    // 3. Insertar en titulos_temarios_cuestionarios
    $stmtRelacion = $conexion->prepare("
        INSERT INTO titulos_temarios_cuestionarios (ID_Titulo, cuestionario_id) 
        VALUES (?, ?)
    ");
    $stmtRelacion->bind_param("ii", $idTitulo, $cuestionarioId);

    if (!$stmtRelacion->execute()) {
        throw new Exception("Error al relacionar título con cuestionario");
    }
    $stmtRelacion->close();

    // Confirmar transacción
    $conexion->commit();

    echo json_encode([
        'success' => true,
        'message' => 'Título creado y asociado al cuestionario correctamente',
        'id_titulo' => $idTitulo
    ]);
} catch (Exception $e) {
    $conexion->rollback();
    error_log("Error en controlador: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error al procesar la solicitud. Por favor intente nuevamente.'
    ]);

    // Cerrar statements si están definidos
    if (isset($stmtInsertTitulo)) $stmtInsertTitulo->close();
    if (isset($stmtCheckCuestionario)) $stmtCheckCuestionario->close();
    if (isset($stmtRelacion)) $stmtRelacion->close();
}

$conexion->close();